#!/bin/bash

#########################
## ALCASAR replication ##
##        start        ##
#########################
# The script is designed to start a connection to a remote.

# Constants
readonly ALCASAR_PWD="/root/ALCASAR-passwords.txt"

# Dynamically generated constants
DB_ROOT_PWD="$(grep db_root "$ALCASAR_PWD" | cut -d '=' -f 2-)"
readonly DB_ROOT_PWD;

# Variables
remote_to_start=""


# Check script args
# $@: script args
check_args() {
	# Parse args
	args="$(getopt --longoptions "all,name:,help" --options "a,n:,h" -- "$@")"

	# Reset script args list
	eval set -- "$args"

	# Print help
	if [ "$#" -eq 1 ]
	then
		usage
		return 1
	fi

	# Loop over all args
	while true
	do
		case "$1" in
			--all | -a)
				remote_to_start="ALL"
				break
				;;
			--name | -n)
				remote_to_start="$2"
				shift
				break
				;;
			--help | -h)
				usage
				return 2
				;;
			--)
				# End of args
				break
				;;
			*)
				echo "error: unknown $1" >&2
				return 3
				break
				;;
		esac
		shift
	done
}

# Execute SQL query
# $1: query
# $2: user (default: root)
# $3: password (default: root pwd)
# $4: host (default: localhost)
# $5: port (default: 3306)
exec_query() {
	# Check args
	if [ $# -lt 1 ]
	then
		echo "usage: $0 \"SQL query\" <DB user> <DB password> <SQL server address> <SQL server port>"
		return 2
	fi

	# Execute the query
	/usr/bin/mariadb --user="root" --password="$DB_ROOT_PWD" --execute="$1"
}

# Start all or specific replication with remotes
start_remote() {
	# Check args
	if [ $# -ne 1 ]
	then
		echo "error: need to pass remote name"
		return 1
	fi

	name="$1"
	attributes="$(/usr/local/bin/alcasar-replication-list.sh --name=$name)"
	if echo "$attributes" | grep -q "Master_Host";
	then
		echo "Starting replication with '$name'"
		exec_query "START REPLICA '$name'" || return
	else
		echo "Remote $name doesn't exists !"
	fi
}

# Print help message
usage() {
	echo "usage: $0 OPTIONS"
	echo
	echo "OPTIONS"
	echo "	--all, -a"
	echo "		start all connections"
	echo "	--name=NAME, -n NAME"
	echo "		remote name connection to start"
	echo "	--help, -h"
	echo "		print this help message"
}

# Main
check_args "$@" || exit

# Stop all or specific remote
if [ "$remote_to_start" == "ALL" ]
then
	# Get remotes names from database
	remotes_names=$(/usr/local/bin/alcasar-replication-list.sh --all | grep "Connection_name" | cut -d ':' -f 2-)
	# Loop over names
	for name in $remotes_names
	do
		start_remote "$name"
	done
else
	start_remote "$remote_to_start"
fi
