#!/bin/bash

#########################
## ALCASAR replication ##
##        delete       ##
#########################
# The script is designed to delete a connection to a remote.

# Constants
readonly ALCASAR_PWD="/root/ALCASAR-passwords.txt"

# Dynamically generated constants
DB_ROOT_PWD="$(grep db_root "$ALCASAR_PWD" | cut -d '=' -f 2-)"
readonly DB_ROOT_PWD;

# Variables
remote_to_delete=""

# Check script args
# $@: script args
check_args() {
	# Parse args
	args="$(getopt --longoptions "all,name:,help" --options "a,n:,h" -- "$@")"

	# Reset script args list
	eval set -- "$args"

	# Print help
	if [ "$#" -eq 1 ]
	then
		usage
		return 1
	fi

	# Loop over all args
	while true
	do
		case "$1" in
			--all | -a)
				echo "Removing all connections."
				remote_to_delete="ALL"
				break
				;;
			--name | -n)
				echo "Removing connection to $2."
				remote_to_delete="$2"
				shift
				break
				;;
			--help | -h)
				usage
				return 2
				;;
			--)
				# End of args
				break
				;;
			*)
				echo "error: unknown $1" >&2
				return 3
				break
				;;
		esac
		shift
	done
}

# Execute SQL query
# $1: query
exec_query() {
	# Check args
	if [ $# -lt 1 ]
	then
		echo "usage: $0 \"SQL query\" <DB user> <DB password> <SQL server address> <SQL server port>"
		return 2
	fi

	# Execute the query
	/usr/bin/mariadb --user="root" --password="$DB_ROOT_PWD" --execute="$1"
}

# Delete replication with a remote
# $1: remote name to delete
delete_remote() {
	# Check args
	if [ $# -ne 1 ]
	then
		echo "error: need to pass remote name"
		return 1
	fi
	name="$1"
	# Check for replication
	attributes="$(/usr/local/bin/alcasar-replication-list.sh --name=$name)"
	if ! echo "$attributes" | grep -q "Master_Host"; then
		echo "no repliqua to $name"
		return 1
	fi
	echo "Stop slave in '$name'..."
	exec_query "STOP SLAVE '$name'" || return
	echo "Resetting replication with '$name'..."
	exec_query "RESET REPLICA '$name' ALL" || return
	service_file="replication-$name.service"
	service_path="/etc/systemd/system/$service_file"
	if [ -f "$service_path" ]; then # for slaves only
		echo "Deleting SSH tunnel with '$name'..."
		systemctl stop "$service_file"
		systemctl disable "$service_file"
		port="$(grep "ExecStart" "$service_path" | cut -d ' ' -f 9)"
		ip="$(grep "ExecStart" "$service_path" | cut -d ' ' -f 14 | cut -d '@' -f2)"
		rm "$service_path"
		echo "Disabling outbound connection to remote SSH ($ip:$port)"
		sed -i "/^REPLICATION_TO=/s/$ip:$port,//" /usr/local/etc/alcasar.conf
		/usr/local/bin/alcasar-iptables.sh
	fi
}

# Print help message
usage() {
	echo "usage: $0 OPTIONS"
	echo
	echo "OPTIONS"
	echo "	--all, -a"
	echo "		delete all connections"
	echo "	--name=NAME, -n NAME"
	echo "		remote name connection to delete"
	echo "	--help, -h"
	echo "		print this help message"
}

# Main
check_args "$@" || exit

# Delete all or specific remote
if [ "$remote_to_delete" == "ALL" ]
then
	# Get remotes names from database
	remotes_names=$(/usr/local/bin/alcasar-replication-list.sh --all | grep "Connection_name" | cut -d ':' -f 2-)

	# Loop over names
	for name in $remotes_names
	do
		delete_remote "$name"
	done
else
	delete_remote "$remote_to_delete"
fi
