#!/bin/bash
#
# $Id: alcasar-importcert.sh 3300 2025-09-23 16:11:12Z rexy $
#
# alcasar-importcert.sh
# by Raphaël, Hugo, Clément, Bettyna & rexy
#
# This script is distributed under the Gnu General Public License (GPL)
#
# Script permettant
# - d'importer des certificats sur Alcasar
# - de revenir au certificat par default
#
# This script allows
# - to import a certificate in Alcasar
# - to go back to the default certificate

SED="/bin/sed -ri"
DIR_CERT="/etc/pki/tls"
CONF_FILE="/usr/local/etc/alcasar.conf"
PRIVATE_IP_MASK=`grep ^PRIVATE_IP= $CONF_FILE|cut -d"=" -f2`
PRIVATE_IP=`echo $PRIVATE_IP_MASK | cut -d"/" -f1`

usage="Usage: alcasar-importcert.sh -i /path/to/certificate.crt -k /path/to/privatekey.key [-c /path/to/serverchain.crt]\n       alcasar-importcert.sh -d (restore default certificate)"
nb_args=$#
arg1=$1

function defaultCert()
{
	mv -f $DIR_CERT/certs/alcasar.crt.old $DIR_CERT/certs/alcasar.crt
	mv -f $DIR_CERT/private/alcasar.key.old $DIR_CERT/private/alcasar.key
	if [ -f $DIR_CERT/certs/server-chain.pem.old ]
	then
		mv $DIR_CERT/certs/server-chain.pem.old $DIR_CERT/certs/server-chain.pem
	fi
	(cat $DIR_CERT/private/alcasar.key; echo; cat $DIR_CERT/certs/alcasar.crt) > $DIR_CERT/private/alcasar.pem
	chown root:apache $DIR_CERT/private/alcasar.pem
	chmod 750 $DIR_CERT/private/alcasar.pem
}

function domainName() # change the domain name in the conf files
{
	fqdn=$(openssl x509 -noout -subject -nameopt multiline -in $DIR_CERT/certs/alcasar.crt | grep commonName|cut -d"=" -f2|tr -d ' ')
	#check if there is a wildcard in $fqdn
	if [[ $fqdn == *"*"* ]];
	then
		hostname="alcasar"
		fqdn=${fqdn/"*"/$hostname}
	else
		hostname=$(echo $fqdn | cut -d'.' -f1)
	fi
	domain=$(echo $fqdn | cut -d'.' -f2-)
	echo "fqdn=$fqdn hostname=$hostname domain=$domain"
	#check fqdn format
	if [[ "$fqdn" != "" && "$domain" != "" ]]; then
		$SED "s/^HOSTNAME=.*/HOSTNAME=$hostname/g" /usr/local/etc/alcasar.conf
		$SED "s/^DOMAIN=.*/DOMAIN=$domain/g" /usr/local/etc/alcasar.conf
		/usr/local/bin/alcasar-conf.sh --apply
	fi
}

function certImport()
{
	if [ ! -f "$DIR_CERT/certs/alcasar.crt.old" ]
	then
		echo "Backup of old cert (alcasar.crt)"
		mv $DIR_CERT/certs/alcasar.crt $DIR_CERT/certs/alcasar.crt.old
	fi
	if [ ! -f "$DIR_CERT/private/alcasar.key.old" ]
	then
		echo "Backup of old private key (alcasar.key)"
		mv $DIR_CERT/private/alcasar.key $DIR_CERT/private/alcasar.key.old
	fi
	cp $cert $DIR_CERT/certs/alcasar.crt
	cp $key $DIR_CERT/private/alcasar.key
	(cat $DIR_CERT/private/alcasar.key; echo; cat $DIR_CERT/certs/alcasar.crt) > $DIR_CERT/private/alcasar.pem
	chown root:apache $DIR_CERT/certs/alcasar.crt
	chown root:apache $DIR_CERT/private/alcasar.key
	chown root:apache $DIR_CERT/private/alcasar.pem
	chmod 750 $DIR_CERT/certs/alcasar.crt
	chmod 750 $DIR_CERT/private/alcasar.key
	chmod 750 $DIR_CERT/private/alcasar.pem
	if [ "$sc" != "" ]
	then
		echo "cert-chain exists"
		if [ ! -f "$DIR_CERT/certs/server-chain.pem.old" ]
		then
			echo "Backup of old cert-chain (server-chain.pem)"
			mv $DIR_CERT/certs/server-chain.pem $DIR_CERT/certs/server-chain.pem.old
		fi
		cp $sc $DIR_CERT/certs/server-chain.pem
		chown root:apache $DIR_CERT/certs/server-chain.pem
		chmod 750 $DIR_CERT/certs/server-chain.pem
	fi
}


if [ $nb_args -eq 0 ]
then
	echo -e "$usage"
	exit 1
fi

case $arg1 in
	-\? | -h* | --h*)
		echo -e "$usage"
		exit 0
		;;
	-i)
		arg3=$3
		arg5=$5
		cert=$2
		key=$4
		sc=$6

		if [ "$cert" == "" ] || [ "$key" == "" ]
		then
			echo -e "$usage"
			exit 1
		fi

		if [ ! -f "$cert" ] || [ ! -f "$key" ]
		then
			echo "Certificate and/or private key not found"
			exit 1
		fi

		if [ ${cert: -4} != ".crt" ] && [ ${cert: -4} != ".cer" ]
		then
			echo "Invalid certificate file"
			exit 1
		fi

		if [ ${key: -4} != ".key" ]
		then
			echo "Invalid private key"
			exit 1
		fi

		if [ "$arg5" != "-c" ] || [ -z "$sc" ]
		then
			echo "No server-chain given"
			echo "Importing certificate $cert with private key $key"
			sc=""
		else
			if [ ! -f "$sc" ]
			then
				echo "Server-chain certificate not found"
				exit 1
			fi
			if [ ${sc: -4} != ".crt" ] && [ ${sc: -4} != ".cer" ] && [ ${sc: -4} != ".pem" ]
			then
				echo "Invalid server-chain certificate file"
				exit 1
			fi
			echo "Importing certificate $cert with private key $key and server-chain $sc"
		fi
		certImport 
		domainName
		;;
	-d)
		if [ -f "/etc/pki/tls/certs/alcasar.crt.old" -a -f "/etc/pki/tls/private/alcasar.key.old" ]
		then
			echo "Restoring default certificate"
			defaultCert
			domainName
			# Reset Letsencrypt file
			cat <<EOF > /usr/local/etc/alcasar-letsencrypt
email=
dateIssueRequest=
domainRequest=
challenge=
dateIssued=
dnsapi=
dateNextRenewal=
EOF
		else echo "No default cert found"
		fi
		;;
	*)
		echo -e "$usage"
		;;
esac
