#!/bin/bash
# $Id: alcasar-dns-local.sh 3238 2024-12-02 22:38:59Z rexy $

# alcasar-dns-local.sh
# by Rexy - 3abtux
# This script is distributed under the Gnu General Public License (GPL)

# active ou desactive la redirection du service DNS sur le réseau de consultation
# enable or disable the redirector of internal DNS service on consultation LAN

SED="/bin/sed -i"

ALCASAR_CONF_FILE="/usr/local/etc/alcasar.conf"
LOCAL_DOMAIN_CONF_FILE="/etc/unbound/conf.d/common/local-forward/dns-redirector.conf"
LOCAL_HOSTNAME_FILE="/etc/hosts"

# define DNS parameters (LAN side)
INT_DNS_DOMAIN=`grep ^DOMAIN $ALCASAR_CONF_FILE|cut -d"=" -f2`
INT_DNS_HOST=`grep ^HOSTNAME $ALCASAR_CONF_FILE|cut -d"=" -f2`
INT_DNS_IP_MASK=`grep ^PRIVATE_IP $ALCASAR_CONF_FILE|cut -d"=" -f2`
INT_DNS_IP=`grep ^PRIVATE_IP $ALCASAR_CONF_FILE|cut -d"=" -f2|cut -d"/" -f1`
INTIF=`grep ^INTIF $ALCASAR_CONF_FILE|cut -d"=" -f2`
INT_DNS_ACTIVE=`grep INT_DNS_ACTIVE $ALCASAR_CONF_FILE|cut -d"=" -f2`
LOCAL_DNS_FILE="/etc/unbound/conf.d/common/local-dns/$INTIF.conf"

usage="Usage: alcasar-dns-local.sh {--on | -on} | {--off | -off} | {--add | -add} ip domain | {--del | -del} ip domain | {--reload | -reload}"
nb_args=$#
args=$1
if [ $nb_args -eq 0 ]
then
	echo "$usage"
	exit 1
fi

function restart_dns(){
	for dns in unbound unbound-blacklist unbound-whitelist unbound-blackhole
	do
		echo "Restarting $dns. Please wait..."
		systemctl restart $dns
	done
}

function hosts_to_unbound(){  # configure the unbound conf file with local host names resolution
		cat << EOF > $LOCAL_DNS_FILE
server:
	local-data: "$INT_DNS_HOST.$INT_DNS_DOMAIN IN A $INT_DNS_IP"
	local-data-ptr: "$INT_DNS_IP $INT_DNS_HOST.$INT_DNS_DOMAIN"
EOF
	while read -r line
	do
		ip_address=$(echo $line | awk '{ print $1 }')
		domain=$(echo $line | awk '{ print $2 }')
		if ! echo $line | grep -E -q "^([0-9\.\t ]+alcasar( |$)|127\.0\.0)"
		then
			echo -e "\tlocal-data: \"$domain.$INT_DNS_DOMAIN IN A $ip_address\"" >> $LOCAL_DNS_FILE
			echo -e "\tlocal-data-ptr: \"$ip_address $domain.$INT_DNS_DOMAIN\"" >> $LOCAL_DNS_FILE
		fi
	done < $LOCAL_HOSTNAME_FILE
	if [ $INT_DNS_DOMAIN == "lan" ]
	then
		echo -e "\tlocal-zone: \"$INT_DNS_DOMAIN\" static" >> $LOCAL_DNS_FILE  # Don't forward this local zone
	fi
}

case $args in
	-\? | -h | --h)
		echo "$usage"
		exit 0
		;;
	--add|-add) # add a local host resolution
		if [ $nb_args -ne 3 ]
		then
			echo "$usage"
			exit 1
		else
			# removing if already exists
			$SED "/^$2[ \t]*$3/d" $LOCAL_HOSTNAME_FILE
			# adding to the hosts file
			echo -e "$2\t$3" >> $LOCAL_HOSTNAME_FILE
			hosts_to_unbound
			restart_dns
		fi
		;;
	--del|-del) # remove a local host resolution
		if [ $nb_args -ne 3 ]
		then
			echo "$usage"
			exit 1
		else
			$SED "/^$2[ \t]*$3/d" $LOCAL_HOSTNAME_FILE
			hosts_to_unbound
			restart_dns
		fi
		;;
	--reload|-reload)
			hosts_to_unbound
			restart_dns
		;;
	--hosts_to_unbound|-hosts_to_unbound)
			hosts_to_unbound
		;;
	--off|-off) # disable DNS redirector
		rm -f $LOCAL_DOMAIN_CONF_FILE
		$SED "s?^INT_DNS_ACTIVE.*?INT_DNS_ACTIVE=off?g" $ALCASAR_CONF_FILE
		restart_dns
		/usr/local/bin/alcasar-iptables.sh
		;;
	--off-without-restart|-off-without-restart) # disable DNS redirector
		rm -f $LOCAL_DOMAIN_CONF_FILE
		$SED "s?^INT_DNS_ACTIVE.*?INT_DNS_ACTIVE=off?g" $ALCASAR_CONF_FILE
		/usr/local/bin/alcasar-iptables.sh
		;;
	--on|-on) # enable DNS redirector
		cat > $LOCAL_DOMAIN_CONF_FILE << EOF
server:
    local-zone: "$INT_DNS_DOMAIN." transparent
forward-zone:
	name: "$INT_DNS_DOMAIN."
	forward-addr: $INT_DNS_IP
EOF
		$SED "s?^INT_DNS_ACTIVE.*?INT_DNS_ACTIVE=on?g" $ALCASAR_CONF_FILE
		restart_dns
		/usr/local/bin/alcasar-iptables.sh
		;;
	--on-without-restart|-on-without-restart) # enable DNS redirector
		cat > $LOCAL_DOMAIN_CONF_FILE << EOF
server:
    local-zone: "$INT_DNS_DOMAIN." transparent
forward-zone:
	name: "$INT_DNS_DOMAIN."
	forward-addr: $INT_DNS_IP
EOF
		$SED "s?^INT_DNS_ACTIVE.*?INT_DNS_ACTIVE=on?g" $ALCASAR_CONF_FILE
		/usr/local/bin/alcasar-iptables.sh
		;;
	*)
		echo "Argument inconnu : $1";
		echo "$usage"
		exit 1
		;;
esac
