#!/bin/bash
# $Id: alcasar-watchdog.sh 3227 2024-10-08 09:35:31Z rexy $

# alcasar-watchdog.sh
# by Rexy
# This script is distributed under the Gnu General Public License (GPL)
# - Ce script prévient les usagers de l'indisponibilité de l'accès Internet
# - Il déconnecte les usagers dont les équipements réseau ne répondent plus (leur onglet 'status.php' a été fermé)
# - Il deconnecte les usagers dont les adresses MAC sont usurpées
#
# - This script tells users that Internet access is down
# - It logs out users whose PCs are quiet (their status tab is closed)
# - It logs out users whose MAC address is used by other systems (usurped)

export LC_ALL=C
CONF_FILE="/usr/local/etc/alcasar.conf"
EXTIF=`grep ^EXTIF= $CONF_FILE|cut -d"=" -f2`			# EXTernal InterFace
INTIF=`grep ^INTIF= $CONF_FILE|cut -d"=" -f2`			# INTernal InterFace
private_ip_mask=`grep ^PRIVATE_IP= $CONF_FILE|cut -d"=" -f2`
private_ip_mask=${private_ip_mask:=192.168.182.1/24}
PRIVATE_IP=`echo $private_ip_mask |cut -d"/" -f1`
PRIVATE_IP=${PRIVATE_IP:=192.168.182.1}
MULTIWAN=`grep ^MULTIWAN= $CONF_FILE|cut -d"=" -f2`
current_users_file="/tmp/current_users.txt"		# file containing active users with their "status.php" tab open
DIR_WEB="/var/www/html"
Index_Page="$DIR_WEB/index.php"
IPTABLES="/sbin/iptables"
TUNIF="tun0"							# listen device for chilli daemon
OLDIFS=$IFS
IFS=$'\n'

function lan_down_alert ()
# users are redirected on ALCASAR IP address if a LAN problem is detected
{
	case $LAN_DOWN in
	"1")
		logger -t alcasar-watchdog "$EXTIF (WAN card) link down"
		echo "$EXTIF (WAN card) link down"
		/bin/sed -i "s?diagnostic =.*?diagnostic = \"$EXTIF (WAN card) link down\";?g" $Index_Page
		;;
	"2")
		logger -t alcasar-watchdog "can't contact the default router"
		echo "can't contact the default router"
		/bin/sed -i "s?diagnostic =.*?diagnostic = \"can't contact the default router\";?g" $Index_Page
		;;
	"3")
		logger -t alcasar-watchdog "can't resolv DNS queries"
		echo "can't resolv DNS queries"
		/bin/sed -i "s?diagnostic =.*?diagnostic = \"can't resolv DNS queries\";?g" $Index_Page
		;;
	esac
	net_pb=`grep "network_pb = true;" $Index_Page|wc -l`
	if [ $net_pb = "0" ] # if previously up
		then
		/bin/sed -i "s?^\$network_pb.*?\$network_pb = true;?g" $Index_Page
		$IPTABLES -I PREROUTING -t nat -i $TUNIF -p udp --dport domain -j REDIRECT --to-port 56
	fi
}

function lan_test ()
# LAN connectiivity testing
{
	watchdog_process=`ps -C alcasar-watchdog.sh|wc -l`
	if [[ $(expr $watchdog_process) -gt 3 ]]
		then
		echo "ALCASAR watchdog is already running"
		exit 0
	fi
	# EXTIF testing
	LAN_DOWN="0"
	if [ `/sbin/ip link | grep $EXTIF|grep "NO-CARRIER" | wc -l` -eq "1" ]
		then
		LAN_DOWN="1"
	fi
	# Default GW testing
	if [ $LAN_DOWN -eq "0" ]
		then
		GW_EXIST=`/sbin/ip route list|grep ^default|wc -l`
		if [ $GW_EXIST -eq "0" ] # no GW defined !
			then
			systemctl restart network
		else
			if [ "$MULTIWAN" == "off" ] || [ "$MULTIWAN" == "Off" ]
				then
				IP_GW_count=`/sbin/ip route list|grep ^default|wc -l`
				if [ $IP_GW_count -ne 1 ]
					then
					systemctl restart network
				fi
				IP_GW=`/sbin/ip route list|grep ^default|cut -d" " -f3`
				arp_reply=`/usr/sbin/arping -I$EXTIF -c1 $IP_GW|grep response|cut -d" " -f2`
				if [ $arp_reply -eq "0" ]
					then
					LAN_DOWN="2"
				fi
			fi
		fi
	fi
	# DNS request testing (twice)
	if [ $LAN_DOWN -eq "0" ]
		then
			dns_reply=`/usr/bin/host -W1 www.free.fr|grep SERVFAIL|wc -l`
			if [ $dns_reply -eq "1" ]
				then
				dns_reply=`/usr/bin/host -W1 www.startpage.com|grep SERVFAIL|wc -l`
				if [ $dns_reply -eq "1" ]
					then LAN_DOWN="3"
				fi
			fi
	fi
	# if LAN pb detected, users are warned
	if [ $LAN_DOWN != "0" ]
		then
			lan_down_alert
	# else switch in normal mode
	else
		echo "Internet access is OK for now"
		net_pb=`grep "network_pb = true;" $Index_Page|wc -l`
		if [ $net_pb != "0" ] # if already down
			then
			/bin/sed -i "s?^\$network_pb.*?\$network_pb = false;?g" $Index_Page
			$IPTABLES -D PREROUTING -t nat -i $TUNIF -p udp --dport domain -j REDIRECT --to-port 56
		fi
	fi
}

usage="Usage: alcasar-watchdog.sh {-lt --lan_test | --disconnect-permanent-users}"
case $1 in
	-\? | -h* | --h*)
		echo "$usage"
		exit 0
	;;
	-lt | --lan_test)
		lan_test
		exit 0
	;;
	--disconnect-permanent-users)
		/bin/sed -i '/PERM/d' $current_users_file
		exit 0
	;;
	*)
		lan_test
		# We disconnect inactive users (its means that their 'status.php' tab has been closed --> their ip address isn't in $current_users_file)
		# process each equipment known by chilli
		for system in `/usr/sbin/chilli_query list | grep -v "0\.0\.0\.0"`
		do
			active_ip=`echo $system |cut -d" " -f2`
			active_session=`echo $system |cut -d" " -f5`
			active_mac=`echo $system | cut -d" " -f1`
			active_user=`echo $system |cut -d" " -f6`
			# We check if the user isn't an auth @MAC and if he is still connected
			if [ "$active_user" != "$active_mac" ] && [ $(expr $active_session) -eq 1 ]; then
				if [ -e $current_users_file ]; then
					# We check if user @IP is in 'current_users.txt'
					cmp_user_ok=$(cat $current_users_file | awk -F':' "\$1 == \"$active_ip\" {print \$2}")
					# If not we disconnect this user.
					if [ -z "$cmp_user_ok" ]; then
						logger -t alcasar-watchdog "$active_ip ($active_mac) doesn't contact ALCASAR any more. We disconnects the user ($active_user)."
						/usr/sbin/chilli_query logout $active_mac
					elif [ "$cmp_user_ok" == "TEMP" ]; then
						# Remove the user's IP from 'current_users.txt'. Every user status page need to insert their @IP everytime to prove their connectivity.
						# We don't disconnect when $cmp_user_ok == "PERM" (status page not needed)
						sed -i "/^$active_ip:$cmp_user_ok\$/d" $current_users_file
					fi
				else # "current_user.txt" does not exists. We disconnect every users.
					logger -t alcasar-watchdog "The file /tmp/current_users.txt doesn't' exist. We disconnects the user $active_user"
					/usr/sbin/chilli_query logout $active_mac
				fi
			fi
			# IP usurpation test : process only equipment with an authenticated user
			if [[ $(expr $active_session) -eq 1 ]]
			then
				arp_reply=`/usr/sbin/arping -b -I$INTIF -s$PRIVATE_IP -c1 -w4 $active_ip|grep -c "Unicast reply"`
				# disconnect users whose equipement is usurped. For example, if there are 2 same @MAC it will make 2 lines in output.
				if [[ $(expr $arp_reply) -gt 1 ]]
					then 
					echo "[$(date +"%Y-%m-%d %H:%M:%S")] : alcasar-watchdog : $active_ip is usurped ($active_mac). Alcasar disconnect the user ($active_user)." >> /var/Save/security/watchdog.log
					logger -t alcasar-watchdog "$active_ip is usurped ($active_mac). Alcasar disconnect the user ($active_user)."
					/usr/sbin/chilli_query logout $active_mac
					chmod 644 /var/Save/security/watchdog.log
				fi
			fi
		done
	;;
esac
IFS=$OLDIFS
