#!/bin/bash

#########################
## ALCASAR replication ##
##      uninstall      ##
#########################
# The script is designed to delete all replication artefacts.

# Constants
readonly PASSWD_FILE="/root/ALCASAR-passwords.txt"
readonly REPL_USER="replication"
readonly REPL_DB_USER="$REPL_USER"
readonly DB_CONF=/etc/my.cnf.d/server.cnf
readonly DB_REPL_CONF=/etc/my.cnf.d/replication.cnf
readonly ALCASAR_CONF=/usr/local/etc/alcasar.conf

# Dynamically generated constants
DB_ROOT_PWD="$(grep db_root "$PASSWD_FILE" | cut -d '=' -f 2-)"
readonly DB_ROOT_PWD


# Execute SQL queries on local server
exec_query() {
	if [ $# -ne 1 ]
	then
		echo "A SQL query must be given." >&2
		return 1
	fi

	# Execute the query
	/usr/bin/mariadb --user=root --password="$DB_ROOT_PWD" --execute="$1"
}

if grep -q "REPLICATION=off" "$ALCASAR_CONF"
then
	echo "error: replication not installed" >&2
	exit 2
fi

# Remove all remotes connected
/usr/local/bin/alcasar-replication-stop.sh --all
/usr/local/bin/alcasar-replication-delete.sh --all

# Delete generated credentials for system user
sed -i "/# Replication/d" "$PASSWD_FILE"
sed -i "/^replication/d" "$PASSWD_FILE"

# Delete generated credentials for database user
sed -i "/# Database replication/d" "$PASSWD_FILE"
sed -i "/^db_replication/d" "$PASSWD_FILE"

# Delete database replication user
echo "Revoking '$REPL_DB_USER' privileges..."
exec_query "REVOKE ALL PRIVILEGES, GRANT OPTION FROM '$REPL_DB_USER'@'%'" || exit
exec_query "FLUSH PRIVILEGES" || exit
echo "Deleting '$REPL_DB_USER' user from database..."
exec_query "DROP USER IF EXISTS '$REPL_DB_USER'@'%'" || exit

# Disable binary logging
echo "Disabling binary logging..."
rm -f "$DB_REPL_CONF"
exec_query "RESET MASTER" || exit

# Unlisten on localhost
sed -i "s?.*skip-networking.*?skip-networking?" "$DB_CONF"
sed -i "s?^bind-address=.*?#bind-address=127.0.0.1?" "$DB_CONF"

# Apply binary logging
echo "Restarting MariaDB..."
/usr/bin/systemctl restart mariadb.service

# User for SSH tunneling
echo "Dropping replication user..."
/usr/bin/pkill -u "$REPL_USER"
/usr/sbin/userdel -r "$REPL_USER"

# Reset REPLICATION config attributes in conf file
sed -i "/^REPLICATION=/s/on/off/" "$ALCASAR_CONF"

echo "Resetting outbound firewall rules..."
sed -i "/^REPLICATION_TO=/s/=.*$/=/" "$ALCASAR_CONF"
/usr/local/bin/alcasar-iptables.sh

echo "Database replication succesfully uninstalled."
