#!/bin/bash

#########################
## ALCASAR replication ##
##         stop        ##
#########################
# The script is designed to stop a connection to a remote.

# Constants
readonly ALCASAR_PWD="/root/ALCASAR-passwords.txt"

# Dynamically generated constants
DB_ROOT_PWD="$(grep db_root "$ALCASAR_PWD" | cut -d '=' -f 2-)"
readonly DB_ROOT_PWD;

# Variables
remote_to_stop=""

# Check script args
# $@: script args
check_args() {
	# Parse args
	args="$(getopt --longoptions "all,name:,help" --options "a,n:,h" -- "$@")"

	# Reset script args list
	eval set -- "$args"

	# Print help
	if [ "$#" -eq 1 ]
	then
		usage
		return 1
	fi

	# Loop over all args
	while true
	do
		case "$1" in
			--all | -a)
				echo "Stopping all connections."
				remote_to_stop="ALL"
				break
				;;
			--name | -n)
				echo "Stopping connection to $2."
				remote_to_stop="$2"
				shift
				break
				;;
			--help | -h)
				usage
				return 2
				;;
			--)
				# End of args
				break
				;;
			*)
				echo "error: unknown $1" >&2
				return 3
				break
				;;
		esac
		shift
	done
}

# Execute SQL query
# $1: query
# $2: user (default: root)
# $3: password (default: root pwd)
# $4: host (default: localhost)
# $5: port (default: 3306)
exec_query() {
	# Check args
	if [ $# -lt 1 ]
	then
		echo "usage: $0 \"SQL query\" <DB user> <DB password> <SQL server address> <SQL server port>"
		return 2
	fi

	# Execute the query
	/usr/bin/mariadb --user="root" --password="$DB_ROOT_PWD" --execute="$1"
}

# Stop all or specific replication with remotes
# $1: remote name to stop
stop_remote() {
	# Check args
	if [ $# -ne 1 ]
	then
		echo "error: need to pass remote name"
		return 1
	fi

	name="$1"
	service_file="/etc/systemd/system/replication-$name.service"

	echo "Stopping replication with '$name'..."
	exec_query "STOP REPLICA '$name'" || return

	# Skip if the connection is received (like on a primary)
	if [ -f "$service_file" ]
	then
		echo "Stopping SSH tunnel with '$name'..."
		/usr/bin/systemctl stop "replication-$name"
		/usr/bin/systemctl disable "replication-$name"

		# Get remote IP and port from its name
		ip="$(/usr/local/bin/alcasar-replication-list.sh --name=$name | grep "Master_Host" | cut -d ' ' -f 2-)"
		port="$(grep "ExecStart" "$service_file" | cut -d ' ' -f 9)"

		echo "Disabling outbound connection to remote SSH..."
		sed -i "/^REPLICATION_TO=/s/$ip:$port,//" /usr/local/etc/alcasar.conf
		/usr/local/bin/alcasar-iptables.sh
	fi
}

# Print help message
usage() {
	echo "usage: $0 OPTIONS"
	echo
	echo "OPTIONS"
	echo "	--all, -a"
	echo "		stop all connections"
	echo "	--name=NAME, -n NAME"
	echo "		remote name connection to stop"
	echo "	--help, -h"
	echo "		print this help message"
}

# Main
check_args "$@" || exit

# Stop all or specific remote
if [ "$remote_to_stop" == "ALL" ]
then
        # Get remotes names from database
        remotes_names=$(/usr/local/bin/alcasar-replication-list.sh --all | grep "Connection_name" | cut -d ':' -f 2-)

        # Loop over names
        for name in $remotes_names
        do
                stop_remote "$name"
        done
else
        stop_remote "$remote_to_stop"
fi
