#!/bin/bash

########################
## SSH key management ##
########################
# The script is designed to setup manage allowed SSH keys on local replication user

# Constants
readonly REPL_USER="replication"
readonly REPL_SSH_AUTHORIZED_KEYS_FILE="/home/$REPL_USER/.ssh/authorized_keys"

# Variables
mode=""
regex=""
file=""

# Check script args
# $@: script args
check_args() {
	# Parse args
	args="$(getopt --longoptions "add,delete,regex:,file:,list,show-pubkey,help" --options "a,d,r:,f:,l,h" -- "$@")"

	# Reset script args list
	eval set -- "$args"

	# Print help
	if [ "$#" -eq 1 ]
	then
		usage
		return 4
	fi

	# Loop over all args
	while true
	do
		case "$1" in
			--add | -a)
				mode="add"
				;;
			--delete | -d)
				mode="delete"
				;;
			--regex | -r)
				regex="$2"
				shift
				;;
			--file | -f)
				file="$2"
				if [ ! -f "$file" ]
				then
					echo "error: $file doesn't exist"
					return 2
				fi
				shift
				;;
			--list | -l)
				mode="list"
				;;
			--show-pubkey)
				mode="show-pubkey"
				;;
			--help | -h)
				usage
				return 3
				;;
			--)
				# End of args
				break
				;;
			*)
				echo "error: unknown $1" >&2
				return 1
				break
				;;
		esac
		shift
	done

	# All fields must be filled
	case "$mode" in
		add)
			# Needed args to be passed
			if [ -z "$file" ]
			then
				echo "error: file argument is missing" >&2
				return 2
			fi
			;;
		delete)
			# Needed args to be passed
			if [ -z "$regex" ]
			then
				echo "error: key hostregex is missing" >&2
				return 2
			fi
			;;
		*)
			;;
	esac
}

# Print help message
usage() {
	echo "usage: $0 ACTION ARGUMENT"
	echo
	echo "	--help, -h"
	echo "		print this help message"
	echo
	echo "ACTION"
	echo "	--add, -a"
	echo "		add a new key"
	echo "	--delete, -d"
	echo "		delete an authorized key"
	echo "	--list, -l"
	echo "		display authorized keys"
	echo
	echo "ARGUMENT"
	echo "	--file=FILE, -f FILE"
	echo "		pubkey file"
	echo "	--regex=REGEX, -r REGEX"
	echo "		regular expression to match"
	echo
	echo "EXAMPLES"
	echo "	$0 --add --file=/tmp/new_id_rsa.pub"
	echo "	$0 --delete --regex=root@remote.local"
}

# Main
check_args "$@" || exit

# Manage key
case "$mode" in
	add)
		echo "Adding new key to authorized keys list..."
		/usr/bin/cat "$file" >> "$REPL_SSH_AUTHORIZED_KEYS_FILE"
		;;
	delete)
		echo "Deleting key from '$regex'..."
		/usr/bin/sed -i "/$regex/d" "$REPL_SSH_AUTHORIZED_KEYS_FILE"
		;;
	list)
		/usr/bin/cat "$REPL_SSH_AUTHORIZED_KEYS_FILE"
		;;
	show-pubkey)
		/usr/bin/cat /root/.ssh/id_*.pub
		;;
	*)
		echo "error: unknown mode" >&2
		return 1
		;;
esac
