#!/bin/bash

# $Id: alcasar-ldap.sh 2731 2019-05-26 21:09:18Z tom.houdayer $

# alcasar-ldap.sh
# by Rexy
# This script is distributed under the Gnu General Public License (GPL)

# activation / désactivation de l'authentification des utilisateurs via un serveur LDAP externe
# enable / disable authentication of users via an extern LDAP server

usage="Usage: alcasar-ldap.sh {--on or -on } | {--off or -off} | --import-cert {certificatePath} | --test [-d]"
SED="/bin/sed -i"
CONF_FILE="/usr/local/etc/alcasar.conf"
LDAP_MODULE="/etc/raddb/mods-available/ldap-alcasar"
OPENLDAP_CONF='/etc/openldap/ldap.conf'
LDAPS_CERT_LOC='/etc/raddb/certs/alcasar-ldaps.crt'
LDAP_SERVER=$(grep '^LDAP_SERVER=' $CONF_FILE | cut -d"=" -f2)                # hostname/IP address of the LDAP server
LDAP_USER=$(grep '^LDAP_USER=' $CONF_FILE | cut -d"=" -f2-)                   # LDAP username used by ALCASAR to read the remote directory
LDAP_PASSWORD=$(grep '^LDAP_PASSWORD=' $CONF_FILE | cut -d"=" -f2-)           # its password
LDAP_BASE=$(grep '^LDAP_BASE=' $CONF_FILE | cut -d"=" -f2-)                   # Where to find the users (cn=**,dc=**,dc=**)
LDAP_UID=$(grep '^LDAP_UID=' $CONF_FILE | cut -d"=" -f2)                      # 'samaccountname' for A.D. - 'UID' for LDAP
LDAP_FILTER=$(grep '^LDAP_FILTER=' $CONF_FILE | cut -d"=" -f2-)               # LDAP filter
LDAP_SSL=$(grep '^LDAP_SSL=' $CONF_FILE | cut -d"=" -f2-)                     # LDAP SSL status
LDAP_CERT_REQUIRED=$(grep '^LDAP_CERT_REQUIRED=' $CONF_FILE | cut -d"=" -f2-) # LDAP SSL certificate verifying

add_ldap_server_to_static_dhcp() {
	if [[ "$LDAP_SERVER" =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]]; then
		ldap_server_ip="$LDAP_SERVER"
	else
		ldap_server_ip=$(dig +short $LDAP_SERVER)
		[ -z "$ldap_server_ip" ] && return 1
	fi

	if [ -z "$(cat /usr/local/etc/alcasar-ethers | awk -v ldap_server_ip="$ldap_server_ip" '($2==ldap_server_ip)')" ]; then
		ldap_server_mac=$(chilli_query list | awk -v ldap_server_ip="$ldap_server_ip" '($2==ldap_server_ip) {print $1}')
		[ -z "$ldap_server_mac" ] && return 1

		echo "$ldap_server_mac $ldap_server_ip" >> /usr/local/etc/alcasar-ethers
		echo "$ldap_server_mac $ldap_server_ip #LDAP Server" >> /usr/local/etc/alcasar-ethers-info
	fi
}

nb_args=$#
args=$1
if [ $nb_args -eq 0 ]; then
	nb_args=1
	args="-h"
fi

case $args in
	-\? | -h* | --h*)
		echo "$usage"
		exit 0
		;;
	--on | -on)
		$SED "s/^LDAP=.*/LDAP=on/g" $CONF_FILE
		if [ "$LDAP_SSL" == 'on' ]; then
			$SED "s/^\tserver =.*/\tserver = \"ldaps:\/\/${LDAP_SERVER//\"/\\\\\\\"}\"/g" $LDAP_MODULE
			$SED "s/^\tport =.*/\tport = 636/g" $LDAP_MODULE
			[ "$LDAP_CERT_REQUIRED" == 'on' ] && require_cert='demand' || require_cert='never'
			$SED "s/^\t\t#?require_cert =.*/\t\trequire_cert = '$require_cert'/g" $LDAP_MODULE
			echo "TLS_REQCERT $require_cert" > $OPENLDAP_CONF
			[ -f "$LDAPS_CERT_LOC" ] && echo "TLS_CACERT $LDAPS_CERT_LOC" >> $OPENLDAP_CONF
		else
			$SED "s/^\tserver =.*/\tserver = \"ldap:\/\/${LDAP_SERVER//\"/\\\\\\\"}\"/g" $LDAP_MODULE
			$SED "s/^\tport =.*/\tport = 389/g" $LDAP_MODULE
			echo  '' > $OPENLDAP_CONF
		fi
		$SED "s/^\tidentity =.*/\tidentity = \"${LDAP_USER//\"/\\\\\\\"}\"/g" $LDAP_MODULE
		$SED "s/^\tpassword =.*/\tpassword = \"${LDAP_PASSWORD//\"/\\\\\\\"}\"/g" $LDAP_MODULE
		$SED "s/^\tbase_dn =.*/\tbase_dn = \"${LDAP_BASE//\"/\\\\\\\"}\"/g" $LDAP_MODULE
		[ -n "$LDAP_FILTER" ] && filter="$LDAP_FILTER" || filter='&'
		$SED "s/^\t\tfilter =.*/\t\tfilter = \"(\&(${LDAP_UID//\"/\\\\\\\"}=%{%{Stripped-User-Name}:-%{User-Name}})(${filter//&/\\&}))\"/g" $LDAP_MODULE
		if [ ! -e /etc/raddb/mods-enabled/ldap ]; then
			ln -s $LDAP_MODULE /etc/raddb/mods-enabled/ldap
		fi
		[ -e /etc/raddb/sites-enabled/alcasar ] && rm /etc/raddb/sites-enabled/alcasar
		ln -s /etc/raddb/sites-available/alcasar-with-ldap /etc/raddb/sites-enabled/alcasar
		add_ldap_server_to_static_dhcp
		/usr/bin/systemctl restart radiusd.service
		;;
	--off | -off)
		$SED "s/^LDAP=.*/LDAP=off/g" $CONF_FILE
		rm -f /etc/raddb/mods-enabled/ldap
		[ -e /etc/raddb/sites-enabled/alcasar ] && rm /etc/raddb/sites-enabled/alcasar
		ln -s /etc/raddb/sites-available/alcasar /etc/raddb/sites-enabled/alcasar
		/usr/bin/systemctl restart radiusd.service
		;;
	--import-cert)
		cert=$2
		[ -z "$cert" ] && echo "$usage" && exit 1

		[ ! -f "$cert" ] && { echo >&2 "ERR: certificate file \"$cert\" not found" ; exit 1; }

		# TODO : convert DER format to PEM ?
		cp -f "$cert" $LDAPS_CERT_LOC
		chown root:radius $LDAPS_CERT_LOC
		chmod 644 $LDAPS_CERT_LOC

		if [ "$LDAP_CERT_REQUIRED" == 'on' ]; then
			domainName=$(openssl x509 -noout -subject -in $LDAPS_CERT_LOC | cut -d' ' -f2- | sed 's@/[A-Za-z]\+=@\n@g' | tac | tr '\n' '.' | sed  's@\.\+$@@')
			if [ "$domainName" != "$LDAP_SERVER" ]; then
				echo 'WARN: the common name of the certificate is different from the server domain name'
			fi
		fi

		$SED "s/^LDAP_SSL=.*/LDAP_SSL=on/g" $CONF_FILE
		$SED "s/^\tserver =.*/\tserver = \"ldaps:\/\/${LDAP_SERVER//\"/\\\\\\\"}\"/g" $LDAP_MODULE
		$SED "s/^\tport =.*/\tport = 636/g" $LDAP_MODULE
		$SED "s@^#\?\t\tca_file =.*@\t\tca_file = $LDAPS_CERT_LOC@g" $LDAP_MODULE
		[ "$LDAP_CERT_REQUIRED" == 'on' ] && require_cert='demand' || require_cert='never'
		$SED "s/^#\?\t\trequire_cert =.*/\t\trequire_cert = '$require_cert'/g" $LDAP_MODULE
		echo -e "TLS_CACERT $LDAPS_CERT_LOC\nTLS_REQCERT $require_cert" > $OPENLDAP_CONF
		/usr/bin/systemctl restart radiusd.service
		;;
	--delete-cert)
		[ -f "$LDAPS_CERT_LOC" ] && rm -f $LDAPS_CERT_LOC
		;;
	--test)
		[ -n "$2" ] && [ "$2" == '-d' ] && debugOpt='-d229'
		command -v ldapsearch &>/dev/null || { echo >&2 -e "ERR: ldapsearch is not installed\nrun 'dnf install openldap-clients'" ; exit 1; }
		if [ "$LDAP_SSL" == 'on' ]; then
			protocol='ldaps'
			[ "$LDAP_CERT_REQUIRED" == 'on' ] && require_cert='demand' || require_cert='never'
			export LDAPTLS_REQCERT="$require_cert"
			[ -f "$LDAPS_CERT_LOC" ] && export LDAPTLS_CACERT="$LDAPS_CERT_LOC"
		else
			protocol='ldap'
		fi
		[ -n "$LDAP_FILTER" ] && filter="$LDAP_FILTER" || filter='&'
		/usr/bin/ldapsearch $debugOpt -LLL -H "$protocol://$LDAP_SERVER" -x -D "$LDAP_USER" -w "$LDAP_PASSWORD" -b "$LDAP_BASE" "(&($LDAP_UID=*)($filter))" 1.1
		;;
	*)
		echo "Argument inconnu : $1";
		echo "$usage"
		exit 1
		;;
esac
