#!/bin/bash
# $Id: alcasar-iptables.sh 3294 2025-07-20 22:26:38Z rexy $
# This script writes the netfilter rules for ALCASAR (see 'alcasar-iptables-bypass.sh' for iptables rules in Bypass mode)
# Rexy - 3abtux - CPN
#
# Reminders
# There are four channels for log :
#	1 tracability of the consultation equipment with The 'Netflow' kernel module (iptables target = NETFLOW);
#	2 protection of ALCASAR with the Ulog group 1 (default group)
#	3 SSH on ALCASAR with the Ulog group 2;
#	4 extern access attempts on ALCASAR with the Ulog group 3.
# The bootps/dhcp (67) port is always open on tun0/INTIF by coova
CONF_FILE="/usr/local/etc/alcasar.conf"
EXTIF=`grep ^EXTIF= $CONF_FILE|cut -d"=" -f2`				# EXTernal InterFace
INTIF=`grep ^INTIF= $CONF_FILE|cut -d"=" -f2`				# INTernal InterFace
TUNIF="tun0"								# listen device for chilli daemon
private_ip_mask=`grep ^PRIVATE_IP= $CONF_FILE|cut -d"=" -f2`
private_ip_mask=${private_ip_mask:=192.168.182.1/24}
PRIVATE_IP=`echo $private_ip_mask | cut -d"/" -f1`			# ALCASAR LAN IP address
private_network=`/bin/ipcalc -n $private_ip_mask|cut -d"=" -f2`		# LAN IP address (ie.: 192.168.182.0)
private_prefix=`/bin/ipcalc -p $private_ip_mask|cut -d"=" -f2`		# LAN prefix (ie. 24)
PRIVATE_NETWORK_MASK=$private_network/$private_prefix			# LAN IP address + prefix (192.168.182.0/24)
public_ip_mask=`grep ^PUBLIC_IP= $CONF_FILE|cut -d"=" -f2`		# ALCASAR WAN IP address
dhcp_on_extif="off"
if [[ "$public_ip_mask" == "dhcp" ]]
then
	dhcp_on_extif="on"
	PTN="\b(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\/([012]?[0-9]|3[0-2])\b"
	public_ip_mask=`ip addr show $EXTIF | grep -o -E $PTN`
fi
PUBLIC_IP=`echo $public_ip_mask | cut -d"/" -f1`
dns1=`grep ^DNS1= $CONF_FILE|cut -d"=" -f2`
dns2=`grep ^DNS2= $CONF_FILE|cut -d"=" -f2`
dns1=${dns1:=208.67.220.220}
dns2=${dns2:=208.67.222.222}
DNSSERVERS="$dns1,$dns2"						# first and second public DNS servers
INT_DNS_IP=`grep INT_DNS_IP $CONF_FILE|cut -d"=" -f2`			# Adresse du serveur DNS interne
INT_DNS_ACTIVE=`grep INT_DNS_ACTIVE $CONF_FILE|cut -d"=" -f2`	# Activation de la redirection DNS interne
BL_IP_CAT="/usr/local/share/iptables-bl-enabled"			# categories files of the BlackListed IP
WL_IP_CAT="/usr/local/share/iptables-wl-enabled"			# categories files of the WhiteListed IP
TMP_users_set_save="/tmp/users_set_save"				# tmp file for backup users set
TMP_set_save="/tmp/ipset_save"						# tmp file for blacklist and whitelist creation
TMP_ip_gw_save="/tmp/ipset_ip_gw_save"				# tmp file for already connected ips
SSH_LAN=`grep ^SSH_LAN= $CONF_FILE|cut -d"=" -f2`			# SSH LAN port
SSH_LAN=${SSH_LAN:=0}
SSH_WAN=`grep ^SSH_WAN= $CONF_FILE|cut -d"=" -f2`		# SSH WAN port
SSH_WAN=${SSH_WAN:=0}
SSH_WAN_ADMIN_FROM=`grep ^SSH_ADMIN_FROM= $CONF_FILE|cut -d"=" -f2|cut -d"/" -f2`
SSH_WAN_ADMIN_FROM=${SSH_WAN_ADMIN_FROM:="0.0.0.0"}
SSH_WAN_ADMIN_FROM=$([ "$SSH_WAN_ADMIN_FROM" == "0.0.0.0" ] && echo "0.0.0.0/0" || echo "$SSH_WAN_ADMIN_FROM" )
SSH_LAN_ADMIN_FROM=`grep ^SSH_ADMIN_FROM= $CONF_FILE|cut -d"=" -f2|cut -d"/" -f1`
SSH_LAN_ADMIN_FROM=${SSH_LAN_ADMIN_FROM:="0.0.0.0"}
SSH_LAN_ADMIN_FROM=$([ "$SSH_LAN_ADMIN_FROM" == "0.0.0.0" ] && echo "$PRIVATE_NETWORK_MASK" || echo "$SSH_LAN_ADMIN_FROM" )
IPTABLES="/sbin/iptables"
REHABILITED_IP="/etc/e2guardian/lists/group1/exceptionsiteiplist"
ALLOWED_SITES="/usr/local/etc/alcasar-uamallowed"		# @IP of external sites/networks allowed for all
MULTIWAN=`grep ^MULTIWAN $CONF_FILE|cut -d"=" -f2`
PROXY=`grep ^PROXY= $CONF_FILE|cut -d"=" -f2`
PROXY_IP=`grep ^PROXY_IP= $CONF_FILE|cut -d"=" -f2`
REPLICATION_TO=`grep ^REPLICATION_TO= $CONF_FILE | cut -d"=" -f2 | tr ',' '\n'`
nb_gw=`grep ^WAN $CONF_FILE|wc -l`
interlan=`grep ^INTERLAN= $CONF_FILE|cut -d"=" -f2`
interlan=${interlan:=off}

# Allow requests to internal DNS if activated
if [ "$INT_DNS_ACTIVE" = "on" ]
then
	DNSSERVERS="$DNSSERVERS,$INT_DNS_IP"
fi

#ipset name list for load_balancing
gw_list="gw0"
if [ "$MULTIWAN" == "on" ] || [ "$MULTIWAN" == "On" ]; then
	for ((i=1 ; i<=$nb_gw ; i++)); do
		gw_list="${gw_list} gw$i"
	done
fi

# Saving SET of connected users if it exists
ipset list not_filtered 1>/dev/null 2>&1
if [ $? -eq 0 ];
then
	ipset save not_filtered > $TMP_users_set_save
	ipset save av >> $TMP_users_set_save
	ipset save av_bl >> $TMP_users_set_save
	ipset save av_wl >> $TMP_users_set_save
	ipset save proto_0 >> $TMP_users_set_save
	ipset save proto_1 >> $TMP_users_set_save
	ipset save proto_2 >> $TMP_users_set_save
	ipset save proto_3 >> $TMP_users_set_save
fi

# Saving all of the already connected IP in order to put them back in the load balancing after
if [ ! -f $TMP_ip_gw_save ];then
	# Save only if alcasar-network.sh --save has not been executed before
	for i in $gw_list;do
		ipset list $i 1>/dev/null 2>&1
		if [ $? -eq 0 ]
		then
			# the cut -d":" -f5 deletes all the lines with a :, i.e all the lines except the members
			ipset list $i | cut -d":" -f5 | sed '/^[[:space:]]*$/d' >> $TMP_ip_gw_save
		fi
	done
fi

# loading of NetFlow probe (ipt_NETFLOW kernel module)
modprobe ipt_NETFLOW destination=127.0.0.1:2055

# Flush all existing rules
$IPTABLES -F
$IPTABLES -t nat -F
$IPTABLES -t mangle -F
$IPTABLES -F INPUT
$IPTABLES -F FORWARD
$IPTABLES -F OUTPUT

# Flush non default rules on filter and nat tables
$IPTABLES -X
$IPTABLES -t nat -X

# Default policies
$IPTABLES -P INPUT DROP
$IPTABLES -P FORWARD DROP
$IPTABLES -P OUTPUT DROP
$IPTABLES -t nat -P PREROUTING ACCEPT
$IPTABLES -t nat -P POSTROUTING ACCEPT
$IPTABLES -t nat -P OUTPUT ACCEPT

######################################################################################
#                         IPSET dynamically loaded/unloaded by coova                 #
# 'not_filtered' :  @IP of users without DNS filtering                               #
# 'av' :            @IP of user with AV (no more used)                               #
# 'av_bl' :         @IP of blacklisted users                                         #
# 'av_wl' :         @IP of whitelisted users                                         #
# 'proto_0' :       @IP of users without protocol filtering                          #
# 'proto_1 :        @IP of users with web ports                                      #
# 'proto_2 :        @IP of users with web/ftp/mail/ssh ports                         #
# 'proto_3 :        @IP of users with custom ports                                   #
#                         IPSET dynamically loaded/unloaded by unbound               #
# 'wl_ip_allowed' : @IP of web sites allowed for WL users                            #
#                         IPSET statically loaded                                    #
# 'bl_ip_blocked' : @IP of sites in selected blacklisted categories                  #
# 'ip_allowed' :    @IP/@network of external allowed systems/networks for all        #
# 'gwX' :           @IP of Gws (to manage load balancing)                            #
# 'profil1_ports' : list of profil1 allowed ports                                    #
# 'profil2_ports' : list of profil2 allowed ports                                    #
# 'custom_ports' :  list of custom allowed ports                                     #
######################################################################################
# destruction de tous les SET / destroy all SET
ipset flush
ipset destroy

######  'bl_ip_blocked' set  ######
# Compute the length
bl_set_length=$(wc -l $BL_IP_CAT/* | awk '{print $1}' | tail -n 1)
# Loading
echo "create bl_ip_blocked hash:net family inet hashsize 1024 maxelem $bl_set_length" > $TMP_set_save
for category in `ls -1 $BL_IP_CAT | cut -d '@' -f1`
do
	cat $BL_IP_CAT/$category >> $TMP_set_save
done
ipset -! restore < $TMP_set_save
rm -f $TMP_set_save
# Removing of rehabilitated ip
for ip in $(cat $REHABILITED_IP)
do
	ipset -q del bl_ip_blocked $ip
done

###### 'ip_allowed' (usefull only for authenticated and filtered users) ######
ipset create ip_allowed hash:net hashsize 1024
nb_uamallowed=`wc -l $ALLOWED_SITES | cut -d" "  -f1`
if [ $nb_uamallowed != "0" ]
then
	while read ip_allowed_line
	do
		site=`echo $ip_allowed_line|cut -d"\"" -f2`
		ipset add ip_allowed $site
	done < $ALLOWED_SITES
fi

###### 'wl_ip_allowed' set ######
# Fixe size as it is dynamically populated by unbound
wl_set_length=65536
# Loading
echo "create wl_ip_allowed hash:net family inet hashsize 1024 maxelem $wl_set_length" > $TMP_set_save
# get ip-wl files from ACC
for category in `ls -1 $WL_IP_CAT |cut -d '@' -f1`
do
	cat $WL_IP_CAT/$category >> $TMP_set_save
done
ipset -! restore < $TMP_set_save
rm -f $TMP_set_save

# Restoring the connected users SETs if available, otherwise creating SETs
if [ -e $TMP_users_set_save ];
then
	ipset -! restore < $TMP_users_set_save
	rm -f $TMP_users_set_save
else
	ipset create not_filtered hash:ip hashsize 1024
	ipset create av hash:ip hashsize 1024
	ipset create av_bl hash:ip hashsize 1024
	ipset create av_wl hash:ip hashsize 1024
	# user set for protocol filtering
	ipset create proto_0 hash:ip hashsize 1024
	ipset create proto_1 hash:ip hashsize 1024
	ipset create proto_2 hash:ip hashsize 1024
	ipset create proto_3 hash:ip hashsize 1024
fi

##### gw0, gw1, gwX : for load balancing ######
for i in $gw_list; do
	ipset create $i hash:ip
done
cat $TMP_ip_gw_save | while read ip; do
	gw_min="gw0"
	weight=`grep ^PUBLIC_WEIGHT= $CONF_FILE | cut -d"=" -f2`
	already=`ipset list $gw_min | grep Number\ of\ entries: | cut -d":" -f2`
	#The *1000 is here to avoid working on floats in bash
	gw_min_value=$((1000 * $already / $weight))
	i=1
	for gw in $gw_list;do
		if [ "$gw" != "gw0" ]; then
			weight=`grep ^WAN$i= $CONF_FILE | awk -F'"' '{ print $2 }' | awk -F ',' '{ print $2 }'`
			already=`ipset list $gw | grep Number\ of\ entries: | cut -d":" -f2`
			value=$((1000 * $already / $weight))
			if [ $value -lt $gw_min_value ]
			then
				gw_min_value=$value
				gw_min=$gw
			fi
			i=$(($i+1))
		fi
	done
	ipset add $gw_min $ip
done
rm -f $TMP_ip_gw_save

###### profil1_ports set ######
# http https
ipset create profil1_ports bitmap:port range 0-65535
for svc_port in 80 443
do
	ipset add profil1_ports $svc_port
done

###### profil2_ports set ######
# ftp ftp-data ssh smtp http pop3 ftps imap https smtps(implicit) smtps(explicit) pop3s imaps smtps(explicit bis)
ipset create profil2_ports bitmap:port range 0-65535
for svc_port in 20 21 22 25 80 110 115 143 443 465 587 993 995 2525
do
	ipset add profil2_ports $svc_port
done

###### custom_ports set ######
ipset create custom_ports bitmap:port range 0-65535
while read svc_line
do
	svc_on=`echo $svc_line|cut -b1`
	if [ $svc_on != "#" ]
	then
		svc_name=`echo $svc_line|cut -d" " -f1`
		svc_port=`echo $svc_line|cut -d" " -f2`
		if [ $svc_name = "icmp" ]
		then
			svc_icmp="on"
		else
			ipset add custom_ports $svc_port
		fi
	fi
done < /usr/local/etc/alcasar-custom-allowed-ports

#############################
#       PREROUTING          #
#############################
# Mark (and log) the direct attempts to E2guardian listen ports in order to REJECT them in INPUT rules
# 8080 = ipset av_bl
$IPTABLES -A PREROUTING -t nat -i $TUNIF -p tcp -d $PRIVATE_IP -m tcp --dport 8080 -j NFLOG --nflog-group 1 --nflog-prefix "RULE direct-proxy -- DENY "
$IPTABLES -A PREROUTING -t mangle -i $TUNIF -d $PRIVATE_IP -p tcp -m tcp --dport 8080 -j MARK --set-mark 1
# 8081 = ipset av_wl + av (to be redefine)
$IPTABLES -A PREROUTING -t nat -i $TUNIF -p tcp -d $PRIVATE_IP -m tcp --dport 8081 -j NFLOG --nflog-group 1 --nflog-prefix "RULE direct-proxy -- DENY "
$IPTABLES -A PREROUTING -t mangle -i $TUNIF -d $PRIVATE_IP -p tcp -m tcp --dport 8081 -j MARK --set-mark 2
# 8443 = tranparent HTTPS for ipsets av_bl + av_wl + av (future version)
#$IPTABLES -A PREROUTING -t nat -i $TUNIF -p tcp -d $PRIVATE_IP -m tcp --dport 8443 -j NFLOG --nflog-group 1 --nflog-prefix "RULE direct-proxy -- DENY "
#$IPTABLES -A PREROUTING -t mangle -i $TUNIF -d $PRIVATE_IP -p tcp -m tcp --dport 8443 -j MARK --set-mark 6

# Mark the direct attempts to DNS ports (UNBOUND) in order to REJECT them in INPUT rules
# 54 = ipset av_bl
$IPTABLES -A PREROUTING -t mangle -i $TUNIF -d $PRIVATE_IP -p tcp --dport 54 -j MARK --set-mark 3
$IPTABLES -A PREROUTING -t mangle -i $TUNIF -d $PRIVATE_IP -p udp --dport 54 -j MARK --set-mark 3
# 55 = ipset av_wl
$IPTABLES -A PREROUTING -t mangle -i $TUNIF -d $PRIVATE_IP -p tcp --dport 55 -j MARK --set-mark 4
$IPTABLES -A PREROUTING -t mangle -i $TUNIF -d $PRIVATE_IP -p udp --dport 55 -j MARK --set-mark 4
# 56 = blackall
$IPTABLES -A PREROUTING -t mangle -i $TUNIF -d $PRIVATE_IP -p tcp --dport 56 -j MARK --set-mark 5
$IPTABLES -A PREROUTING -t mangle -i $TUNIF -d $PRIVATE_IP -p udp --dport 56 -j MARK --set-mark 5

# users DNS redirection
# 54 = ipset av_bl
$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_bl src -p udp --dport domain -j REDIRECT --to-port 54
$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_bl src -p tcp --dport domain -j REDIRECT --to-port 54
# 55 = ipset av_wl
$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_wl src -p udp --dport domain -j REDIRECT --to-port 55
$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_wl src -p tcp --dport domain -j REDIRECT --to-port 55
# 53 = all other users
$IPTABLES -A PREROUTING -t nat -i $TUNIF ! -d $PRIVATE_IP -p udp --dport domain -j REDIRECT --to-port 53
$IPTABLES -A PREROUTING -t nat -i $TUNIF ! -d $PRIVATE_IP -p tcp --dport domain -j REDIRECT --to-port 53

# Redirect HTTP of 'av_bl' users who want blacklist IP to ALCASAR ('access denied' page)
$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_bl src -m set --match-set bl_ip_blocked dst -p tcp --dport http -j REDIRECT --to-port 80

# Redirect HTTP of 'av_wl' users who want IP not in the WL to ALCASAR ('access denied' page)
$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_wl src -m set ! --match-set wl_ip_allowed dst -p tcp --dport http -j REDIRECT --to-port 80

# accounting of "av_bl + av_wl + av"  users (only syn packets). Other protocols are logged in FORWARD by netflow
$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_bl src ! -d $PRIVATE_IP -p tcp --dport http -m conntrack --ctstate NEW -j NFLOG --nflog-group 1 --nflog-prefix "RULE F_http -- ACCEPT "
$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_wl src ! -d $PRIVATE_IP -p tcp --dport http -m conntrack --ctstate NEW -j NFLOG --nflog-group 1 --nflog-prefix "RULE F_http -- ACCEPT "
$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av src ! -d $PRIVATE_IP -p tcp --dport http -m conntrack --ctstate NEW -j NFLOG --nflog-group 1 --nflog-prefix "RULE F_http -- ACCEPT "

# Redirect outbound "av_bl + av_wl +av" users HTTP requests to E2guardian
# 8080 = ipset av_bl
$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_bl src -m set ! --match-set ip_allowed dst ! -d $PRIVATE_IP  -p tcp --dport http -j REDIRECT --to-port 8080
# 8081 = ipset av_wl & av
#$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_wl src ! -d $PRIVATE_IP -p tcp --dport http -j REDIRECT --to-port 8081
#$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av src ! -d $PRIVATE_IP -p tcp --dport http -j REDIRECT --to-port 8081

# Redirect outbound HTTPS requests of av_bl + av_wl + av users to E2Guardian
#$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_bl src -m set ! --match-set ip_allowed dst ! -d $PRIVATE_IP  -p tcp --dport https -j REDIRECT --to-port 8443
#$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av_wl src -m set ! --match-set ip_allowed dst ! -d $PRIVATE_IP  -p tcp --dport https -j REDIRECT --to-port 8443
#$IPTABLES -A PREROUTING -t nat -i $TUNIF -m set --match-set av src -m set ! --match-set ip_allowed dst ! -d $PRIVATE_IP  -p tcp --dport https -j REDIRECT --to-port 8443

# Redirect NTP request in local NTP server
$IPTABLES -A PREROUTING -t nat -i $TUNIF -s $PRIVATE_NETWORK_MASK ! -d $PRIVATE_IP -p udp --dport ntp -j REDIRECT --to-port 123

# Retrieving the mark associated with a gw for each connection
$IPTABLES -A PREROUTING -t mangle -j CONNMARK --restore-mark

if [ "$PROXY" == "on" ] || [ "$PROXY" == "On" ];then
	$IPTABLES -A PREROUTING -t nat -i $TUNIF ! -d $PRIVATE_IP -p tcp -m multiport --dports http,https -j DNAT --to-destination $PROXY_IP
fi

# Marking for load balancing
if [ "$MULTIWAN" == "on" ] || [ "$MULTIWAN" == "On" ]; then
	temp_index=200
	for i in $gw_list; do
		$IPTABLES -A PREROUTING -t mangle -i $TUNIF -m set --match-set $i src -j MARK --set-mark $temp_index
		temp_index=$(($temp_index+1))
	done
fi

#############################
#         INPUT             #
#############################
# Accept all on loopback
$IPTABLES -A INPUT -i lo -j ACCEPT
$IPTABLES -A OUTPUT -o lo -j ACCEPT

# Drop non-standard connections (FIN-URG-PUSH, XMAS, NullScan, SYN-RST and NEW not SYN)
$IPTABLES -A INPUT -p tcp --tcp-flags FIN,URG,PSH FIN,URG,PSH -j DROP
$IPTABLES -A INPUT -p tcp --tcp-flags ALL ALL -j DROP
$IPTABLES -A INPUT -p tcp --tcp-flags ALL NONE -j DROP
$IPTABLES -A INPUT -p tcp --tcp-flags SYN,RST SYN,RST -j DROP
$IPTABLES -A INPUT -p tcp -m tcp ! --syn -m conntrack --ctstate NEW -j DROP

# If configured, DHCP responses are allowed on EXTIF
if [[ "$dhcp_on_extif" == "on" ]]
then
	$IPTABLES -A INPUT -i $EXTIF -p tcp --dport 68 -j ACCEPT
	$IPTABLES -A INPUT -i $EXTIF -p udp --dport 68 -j ACCEPT
fi

# Drop broadcast & multicast on EXTIF to avoid log
$IPTABLES -A INPUT -m addrtype --dst-type BROADCAST,MULTICAST -j DROP

# Conntrack on INPUT
$IPTABLES -A INPUT -m conntrack --ctstate RELATED,ESTABLISHED -j ACCEPT

# Deny direct connections on E2Guardian listen ports. The concerned paquets have been marked and logged in mangle table (PREROUTING)
$IPTABLES -A INPUT -i $TUNIF -p tcp --dport 8080 -m mark --mark 1 -j REJECT --reject-with tcp-reset # av_bl
$IPTABLES -A INPUT -i $TUNIF -p tcp --dport 8081 -m mark --mark 2 -j REJECT --reject-with tcp-reset # av_wl + av
#$IPTABLES -A INPUT -i $TUNIF -p tcp --dport 8443 -m mark --mark 6 -j REJECT --reject-with tcp-reset # av_bl + av_wl + av (future version)

# Allow HTTP/HTTPS connections to E2Guardian
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -p tcp --dport 8080 -m conntrack --ctstate NEW --syn -j ACCEPT
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -p tcp --dport 8081 -m conntrack --ctstate NEW --syn -j ACCEPT
#$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -p tcp --dport 8443 -m conntrack --ctstate NEW --syn -j ACCEPT # (future version)

# Deny direct connections to DNS ports (UNBOUND). The concerned paquets are marked in mangle table (see PREROUTING)
$IPTABLES -A INPUT -i $TUNIF -p udp --dport 54 -m mark --mark 3 -j REJECT --reject-with icmp-port-unreachable
$IPTABLES -A INPUT -i $TUNIF -p tcp --dport 54 -m mark --mark 3 -j REJECT --reject-with tcp-reset
$IPTABLES -A INPUT -i $TUNIF -p udp --dport 55 -m mark --mark 4 -j REJECT --reject-with icmp-port-unreachable
$IPTABLES -A INPUT -i $TUNIF -p tcp --dport 55 -m mark --mark 4 -j REJECT --reject-with tcp-reset
$IPTABLES -A INPUT -i $TUNIF -p udp --dport 56 -m mark --mark 5 -j REJECT --reject-with icmp-port-unreachable
$IPTABLES -A INPUT -i $TUNIF -p tcp --dport 56 -m mark --mark 5 -j REJECT --reject-with tcp-reset

# Allow DNS connections
# ipset = av_bl
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p udp --dport 54 -j ACCEPT
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p tcp --dport 54 -j ACCEPT
# ipset = av_wl
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p udp --dport 55 -j ACCEPT
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p tcp --dport 55 -j ACCEPT
# blackall
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p udp --dport 56 -j ACCEPT
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p tcp --dport 56 -j ACCEPT

# Internal services access
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p udp --dport domain -j ACCEPT	# DNS
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p tcp --dport domain -j ACCEPT	# DNS
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p icmp --icmp-type 8 -j ACCEPT	# Réponse ping # ping responce
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p icmp --icmp-type 0 -j ACCEPT	# Requête  ping # ping request
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p tcp --dport https -j ACCEPT	# Pages d'authentification et MCC # authentication pages and MCC
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p tcp --dport http -j ACCEPT	# Page d'avertissement filtrage # Filtering warning pages
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p tcp --dport 3990:3991 -j ACCEPT	# Requêtes de deconnexion usagers # Users logout requests
$IPTABLES -A INPUT -i $TUNIF -s $PRIVATE_NETWORK_MASK -d $PRIVATE_IP -p udp --dport ntp -j ACCEPT	# Serveur local de temps # local time server

# SSHD server access from LAN (and from WAN if enabled)
if [ $SSH_LAN -gt 0 ]
	then
	$IPTABLES -A INPUT -i $TUNIF -s $SSH_LAN_ADMIN_FROM -d $PRIVATE_IP -p tcp --dport $SSH_LAN -m conntrack --ctstate NEW -j NFLOG --nflog-group 2 --nflog-prefix "RULE ssh-from-LAN -- ACCEPT"
	$IPTABLES -A INPUT -i $TUNIF -s $SSH_LAN_ADMIN_FROM -d $PRIVATE_IP -p tcp --dport $SSH_LAN -j ACCEPT
fi
if [ $SSH_WAN -gt 0 ]
	then
	$IPTABLES -A INPUT -i $EXTIF -s $SSH_WAN_ADMIN_FROM -d $PUBLIC_IP -p tcp --dport $SSH_WAN -m conntrack --ctstate NEW --syn -j NFLOG --nflog-group 2 --nflog-prefix "RULE ssh-from-WAN -- ACCEPT"
	$IPTABLES -A INPUT -i $EXTIF -s $SSH_WAN_ADMIN_FROM -d $PUBLIC_IP -p tcp --dport $SSH_WAN -j ACCEPT
fi

# Here, we add local rules (i.e. VPN from Internet)
if [ -f /usr/local/etc/alcasar-iptables-local.sh ]; then
	. /usr/local/etc/alcasar-iptables-local.sh
fi

# Deny and log on INPUT from the LAN
$IPTABLES -A INPUT -i $TUNIF -m conntrack --ctstate NEW -j NFLOG --nflog-group 3 --nflog-prefix "RULE rej-int -- REJECT "
$IPTABLES -A INPUT -i $TUNIF -p tcp -j REJECT --reject-with tcp-reset
$IPTABLES -A INPUT -i $TUNIF -p udp -j REJECT --reject-with icmp-port-unreachable

# Reject INTIF access (only when chilli is down)
$IPTABLES -A INPUT -i $INTIF -j NFLOG --nflog-group 3 --nflog-prefix "RULE Protect1 -- REJECT "
$IPTABLES -A INPUT -i $INTIF -j REJECT

# On EXTIF, the access attempts are log in channel 2 (we should test --limit option to avoid deny of service)
$IPTABLES -A INPUT -i $EXTIF -m conntrack --ctstate NEW -j NFLOG --nflog-group 3 --nflog-threshold 10 --nflog-prefix "RULE rej-ext -- DROP"

#############################
#        FORWARD            #
#############################

# Deny IPs of the SET bl_ip_blocked for the set av_bl
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set av_bl src -m set --match-set bl_ip_blocked dst -p icmp -j REJECT --reject-with icmp-host-prohibited
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set av_bl src -m set --match-set bl_ip_blocked dst -p udp -j REJECT --reject-with icmp-host-prohibited
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set av_bl src -m set --match-set bl_ip_blocked dst -p tcp -j REJECT --reject-with tcp-reset

# Allow Conntrack
$IPTABLES -A FORWARD -m conntrack --ctstate ESTABLISHED,RELATED -j ACCEPT

# accept ip_allowed (ie : @IP/@network of external allowed systems/networks like DMZ, own servers, etc.)
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set ip_allowed dst -m conntrack --ctstate NEW -j NETFLOW
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set ip_allowed dst -m conntrack --ctstate NEW -j ACCEPT

# protocols filtering for users (profil1 = http, https)
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set proto_1 src -s $PRIVATE_NETWORK_MASK -p tcp -m set ! --match-set profil1_ports dst -m conntrack --ctstate NEW -j REJECT --reject-with tcp-reset
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set proto_1 src -s $PRIVATE_NETWORK_MASK -p udp -m set ! --match-set profil1_ports dst -m conntrack --ctstate NEW -j REJECT --reject-with icmp-port-unreachable

# protocols filtering for users (profil2 = ftp ftp-data ssh smtp http pop3 ftps imap https smtps(implicit) smtps(explicit) pop3s imaps smtps(explicit bis))
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set proto_2 src -s $PRIVATE_NETWORK_MASK -p tcp -m set ! --match-set profil2_ports dst -m conntrack --ctstate NEW -j REJECT --reject-with tcp-reset
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set proto_2 src -s $PRIVATE_NETWORK_MASK -p udp -m set ! --match-set profil2_ports dst -m conntrack --ctstate NEW -j REJECT --reject-with icmp-port-unreachable

# protocols filtering for users (profil 3 : customized with ACC)
if [ "$svc_icmp" != "on" ]
then
	$IPTABLES -A FORWARD -i $TUNIF -m set --match-set proto_3 src -s $PRIVATE_NETWORK_MASK -p icmp -j REJECT --reject-with icmp-proto-unreachable
fi
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set proto_3 src -s $PRIVATE_NETWORK_MASK -p tcp -m set ! --match-set custom_ports dst -m conntrack --ctstate NEW -j REJECT --reject-with tcp-reset
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set proto_3 src -s $PRIVATE_NETWORK_MASK -p udp -m set ! --match-set custom_ports dst -m conntrack --ctstate NEW -j REJECT --reject-with icmp-port-unreachable

# Users are allowed (or not) to access the network between ALCASAR and the Internet router
if [ "$interlan" != "on" ]
then
	$IPTABLES -A FORWARD -i $TUNIF -d $public_ip_mask -j DROP
fi

# Block 'av_wl' users who want IP not in the WL
$IPTABLES -A FORWARD -i $TUNIF -m set --match-set av_wl src -m set ! --match-set wl_ip_allowed dst -j DROP

# Allow forward connections with log
$IPTABLES -A FORWARD -i $TUNIF -s $PRIVATE_NETWORK_MASK -m conntrack --ctstate NEW -j NETFLOW
$IPTABLES -A FORWARD -i $TUNIF -s $PRIVATE_NETWORK_MASK -m conntrack --ctstate NEW -j ACCEPT

#############################
#         OUTPUT            #
#############################
# We let everything out on INTIF/TUNIF
$IPTABLES -A OUTPUT ! -o $EXTIF -j ACCEPT

# Allow DHCP requests on EXTIF if configured
if [[ "$dhcp_on_extif" == "on" ]]
then
	$IPTABLES -A OUTPUT -o $EXTIF -p tcp --dport 67 -j ACCEPT
	$IPTABLES -A OUTPUT -o $EXTIF -p udp --dport 67 -j ACCEPT
fi

# Allow DNS requests to identified DNS servers
$IPTABLES -A OUTPUT -o $EXTIF -d $DNSSERVERS -p udp --dport domain -m conntrack --ctstate NEW -j ACCEPT

# HTTP & HTTPS requests are allowed with netflow log (from E2guardian)
$IPTABLES -A OUTPUT -o $EXTIF -p tcp --dport http -j NETFLOW
$IPTABLES -A OUTPUT -o $EXTIF -p tcp --dport http -j ACCEPT
#$IPTABLES -A OUTPUT -o $EXTIF -p tcp --dport https -j NETFLOW # When E2guardian will be in HTTPS transparent proxy)
$IPTABLES -A OUTPUT -o $EXTIF -p tcp --dport https -j ACCEPT

# RSYNC requests are allowed (update of Toulouse BL)
$IPTABLES -A OUTPUT -o $EXTIF -p tcp --dport rsync -j ACCEPT

# FTP requests are allowed
# modprobe nf_conntrack_ftp # no more needed with kernel > 5.15.85
# $IPTABLES -t raw -A OUTPUT -p tcp --dport ftp -j CT --helper ftp # no more needed with kernel > 5.15.85
$IPTABLES -A OUTPUT -o $EXTIF -p tcp --dport ftp -j ACCEPT
$IPTABLES -A OUTPUT -o $EXTIF -m conntrack --ctstate ESTABLISHED,RELATED -j ACCEPT

# NTP requests are allowed
$IPTABLES -A OUTPUT -o $EXTIF -p udp --dport ntp -j ACCEPT

# ICMP (ping) requests are allowed
$IPTABLES -A OUTPUT -o $EXTIF -p icmp --icmp-type 8 -j ACCEPT

# LDAP requests are allowed
$IPTABLES -A OUTPUT -o $EXTIF -p tcp -m multiport --dports ldap,ldaps -m conntrack --ctstate NEW,ESTABLISHED -j ACCEPT
$IPTABLES -A OUTPUT -o $EXTIF -p udp -m multiport --dports ldap,ldaps -m conntrack --ctstate NEW,ESTABLISHED -j ACCEPT

# Allow outbound connections to primary servers SSH
for conn in $REPLICATION_TO
do
	addr="$(echo $conn | cut -d ':' -f 1)"
	port="$(echo $conn | cut -d ':' -f 2)"
	# Allow to connect to remote SSH
	$IPTABLES -A OUTPUT -o $EXTIF -d "$addr" -p tcp --dport "$port" -j ACCEPT
done

#############################
#       POSTROUTING         #
#############################
# Dynamic NAT on EXTIF
$IPTABLES -A POSTROUTING -t nat -o $EXTIF -j MASQUERADE

# Saving the load balancing mark
$IPTABLES -A POSTROUTING -t mangle -j CONNMARK --save-mark

#############################
#          FAIL2BAN         #
#############################
# Reload Fail2Ban
if systemctl -q is-active fail2ban; then
	/usr/bin/fail2ban-client ping &>/dev/null && /usr/bin/fail2ban-client -q reload &>/dev/null
fi
