#!/bin/bash
#
# Build Mageiar (Mageia + ALCASAR) ISO file
# Authors: Hamza ESSAYEGH (Querdos) - Rexy - Laurent Roux - Aurelien Dubois - Pierre Rivault
#
####################################
###### Variables & functions  ######
####################################

CURRENT_DIR="$(readlink -f "$(dirname $0)")"
LOG_FILE="$CURRENT_DIR/build-image.log"
RESSOURCES="$CURRENT_DIR/ressources"

# Print the given error message and exit 1
function errorExit()
{
    tput bold
    echo -e "ERROR : $1 \nAborting\n" >&2
    echo "$1" > "$LOG_FILE"
    tput sgr0
    exit 1
}
#-------------------------------------------------------------------------------
# Print progress bar. $1 is a multiple of five
function printProgress()
{
    local a
    let "a = $1 / 5"
    let "b = 25 - $a"
    progressStr=$(printf "%${a}s")
    progressStrBlanks=$(printf "%${b}s")
    echo -en "Downloading...$1% [${progressStr// /#}${progressStrBlanks// / }]\r"
}
#-------------------------------------------------------------------------------
function printBold()
{
    tput bold 2>/dev/null
    echo $1
    tput sgr0 2>/dev/null
}

ISO_IN=`ls $RESSOURCES | grep '^Mageia.*iso'`
FULL_PATH_ISO_IN=$CURRENT_DIR/ressources/$ISO_IN
MAGEIA_VERSION=`echo $ISO_IN|cut -d"-" -f2`
ARCH=`echo $ISO_IN |cut -d"-" -f3|cut -d"." -f1`
ALCASAR_TARBALL=`ls $RESSOURCES| grep '^alcasar.*gz'`
ALCASAR_EXTRACTED_DIR=$(echo $ALCASAR_TARBALL | rev | cut -d '.' -f 3- | rev)
ALCASAR_VERSION=`echo $ALCASAR_EXTRACTED_DIR | cut -d"-" -f2`
TMP_DIR=/var/tmp
MAGEIA_NEW_DIR=$TMP_DIR/mageia_new
MAGEIA_OFFICIAL_DIR=$TMP_DIR/mageia_official
coreDir=${MAGEIA_OFFICIAL_DIR}/${ARCH}/media/core
nonFreeDir=${MAGEIA_OFFICIAL_DIR}/${ARCH}/media/nonfree
mediaDirNew=${MAGEIA_NEW_DIR}/${ARCH}/media
coreDirNew=${MAGEIA_NEW_DIR}/${ARCH}/media/core/
nonFreeDirNew=${MAGEIA_NEW_DIR}/${ARCH}/media/nonfree
urpmi_cache=/var/cache/urpmi/rpms

# Drake installer
INSTALLER_DIR="${MAGEIA_NEW_DIR}/${ARCH}/install/alcasar"
EFI_GRUB_CFG="${MAGEIA_NEW_DIR}/boot/grub2/grub.cfg"
ADVERT_DIR="${MAGEIA_NEW_DIR}/${ARCH}/install/extra/advertising"
AUTO_INSTALL_CFG=$TMP_DIR/auto_inst.cfg.pl

# RPMs needed during an install stage
BASE_RPM_LIST="acpi acpid alsa-utils aoss arp-scan basesystem bash-completion coreutils-doc cpupower curl dhcp-client dmraid dnf dnf-plugins-core dosfstools dracut drakx-net-text gpm grub2 grub2-efi grub2-mageia-theme harddrake hdparm hexedit info ldetect lftp lib64alsa-plugins lib64glib-networking-gnutls locales-en locales-fr lsof lvm2 mageia-theme-Default man-db man-pages microcode microcode_ctl mtools ntfs-3g numlock os-prober p11-kit perl-Hal-Cdroms plymouth procmail python3 python3-dbus radeon-firmware ralink-firmware rtlwifi-firmware sharutils shorewall-ipv6 strace sysfsutils tmpwatch tree vim-minimal vnstat xdg-user-dirs-gtk sudo socat mandi mandi-ifw"
ISO_RPM_LIST="cdrkit-genisoimage xorriso rpmtools syslinux"

########################
######    Main    ######
########################

# Some checks
[[ $(whoami) != 'root' ]] && errorExit 'Please run as root'
[[ $ISO_IN == "" ]] && errorExit "No Mageia iso file in $RESSOURCES"
[[ ! -f "$RESSOURCES/auto_inst.cfg.pl_template" ]] && errorExit "No auto_inst.cfg.pl_template file in $RESSOURCES"
[[ $ALCASAR_TARBALL == "" ]] && errorExit "No ALCASAR tarball file in $RESSOURCES"

for directory in $MAGEIA_NEW_DIR $MAGEIA_OFFICIAL_DIR
do
	[ -d $directory ] || mkdir -p $directory
done
printBold "Mageiar will be build with Mageia-$MAGEIA_VERSION-$ARCH and $ALCASAR_EXTRACTED_DIR" 
# Update the system. Keep only RPMs needed to create ALCASAR rpmslist
urpmi --auto-update --auto 2>/dev/null
urpme $ISO_RPM_LIST --auto-orphans --force 2>/dev/null
urpmi $BASE_RPM_LIST --force --auto 2>/dev/null
# Retrieve list of installed RPM (without those that will be installed with the alcasar install script)
RPM_LIST="$(rpm -qa --queryformat "%{NAME}\n"| grep -Ev 'coova-chilli|gammu|lib64gammu8|ipt-netflow|nfdump|wkhtmltopdf' | tr '\n' ' ')"

# Insert list into AUTO_INSTALL_CFG
formattedList=$(echo $RPM_LIST | sed "s/[^ ]*/\'\0\'/g" | sed "s/ /,\n/g")
cp -f $RESSOURCES/auto_inst.cfg.pl_template $AUTO_INSTALL_CFG
insertLineNumber=$(grep -n "'default_packages' => " "$AUTO_INSTALL_CFG" | cut -d ':' -f1)
fileTop=$(head -n "+$insertLineNumber" "$AUTO_INSTALL_CFG")
fileBottom=$(tail -n "+$insertLineNumber" "$AUTO_INSTALL_CFG" | tail -n +2)
cat /dev/null > "$AUTO_INSTALL_CFG"
echo "$fileTop" > "$AUTO_INSTALL_CFG"
echo "$formattedList" >> "$AUTO_INSTALL_CFG"
echo "$fileBottom" >> "$AUTO_INSTALL_CFG"

# Installing tools to create the iso file
printBold "Installing ISO building tools"
#dnf install -y $ISO_RPM_LIST || errorExit "could not install necessary packages"
urpmi $ISO_RPM_LIST --force|| errorExit "could not install necessary packages"

# Mounting the ISO image
printBold "Mounting the image"
if [ `mount |grep $MAGEIA_OFFICIAL_DIR | wc -l` -eq 0 ]; then # if not already mounted
	mount -o ro,loop $RESSOURCES/$ISO_IN $MAGEIA_OFFICIAL_DIR || errorExit "failed mounting $ISO_IN."
fi
sleep 1 # wait for mounting process
[ -d ${MAGEIA_OFFICIAL_DIR}/x86_64 ] || umount $MAGEIA_OFFICIAL_DIR || errorExit "The ISO file isn't in the target architecture ($ARCH)" 

# Copying main files except core and nonfree directories
echo "Extracting base image..."
cd $MAGEIA_OFFICIAL_DIR && \
    tar cf - --exclude=${ARCH}/media . | (cd $MAGEIA_NEW_DIR && tar xf - ) && cd "${CURRENT_DIR}"

# Creating new directories core and nonfree and dir for alcasar stufs
mkdir -p ${MAGEIA_NEW_DIR}/${ARCH}/{media/{core,nonfree},install/alcasar}

# Adding Alcasar image advert to the installer
cp -f $RESSOURCES/install_slideshow/* "$ADVERT_DIR" || errorExit "could not copy custom slideshow to $ADVERT_DIR"

# Create the installed RPM list excluding those brought with alcasar archive (they aren't in repository)
nbInstalled=$(echo $RPM_LIST | wc -w)
total=`echo $RPM_LIST | wc -w`
count=0
# Copying the RPM in core and clearing the plop.idx file
echo "Copying $total RPMS in ISO ..."
for rpm in ${RPM_LIST}; do
	let percent="${count} * 100 / ${total}"
	full_rpm_name=`rpm -q $rpm`
	if [ `ls $urpmi_cache/$full_rpm_name* 2>/dev/null | wc -l` -ne 1 ] ; then  # rpm isn't already in $urpmi_cache
		if [ `ls $coreDir/$full_rpm_name* 2>/dev/null | wc -l` -eq 1 ] ; then  # rpm is in official ISO file
			cp $coreDir/$full_rpm_name* $urpmi_cache/
		else # rpm must be downloaded
			# dnf download --downloaddir $coreDirNew $rpm 1> /dev/null || errorExit "could not download $rpm"
			urpmi --no-install --replacepkgs -q $rpm 2> /dev/null || urpmi --no-install -q --force $rpm 2> /dev/null || echo "could not download $rpm"
		fi
	fi
	printProgress ${percent}
	count=$(expr ${count} + 1)
done
printBold "$count RPMs copied"
# Copying the Alcasar tarball & automatic install option into the ISO
mkdir -p $INSTALLER_DIR || errorExit "could not create directory $INSTALLER_DIR"
cp $RESSOURCES/$ALCASAR_TARBALL $INSTALLER_DIR || errorExit "could not copy $RESSOURCES/$ALCASAR_TARBALL to $INSTALLER_DIR"
cp $AUTO_INSTALL_CFG $INSTALLER_DIR || errorExit "could not copy $AUTO_INSTALL_CFG to $INSTALLER_DIR"

mv $urpmi_cache/*.rpm $coreDirNew
# Generating media info for core
echo "Generating media_info for core..."
genhdlist2 ${coreDirNew} --allow-empty-media --quiet
echo "Generating media_info for nonfree..."
genhdlist2 ${nonFreeDirNew} --allow-empty-media --quiet

# Puting pubkeys in media_info
cp ${coreDir}/media_info/pubkey ${coreDirNew}/media_info/
cp ${nonFreeDir}/media_info/pubkey ${nonFreeDirNew}/media_info/

# Retrieving media.cfg & compssUsers.pl depending on the arch
mkdir ${MAGEIA_NEW_DIR}/${ARCH}/media/media_info
cp ${MAGEIA_OFFICIAL_DIR}/${ARCH}/media/media_info/compssUsers.pl ${MAGEIA_NEW_DIR}/${ARCH}/media/media_info/compssUsers.pl
cp ${MAGEIA_OFFICIAL_DIR}/${ARCH}/media/media_info/media.cfg ${MAGEIA_NEW_DIR}/${ARCH}/media/media_info/media.cfg

# Specify install configuration
#sed -i -e 's*rdz*rdz kickstart=/tmp/media/x86_64/install/alcasar/auto_inst.cfg.pl*g'  ${MAGEIA_NEW_DIR}/isolinux/isolinux.cfg
sed -i -e 's*noiswmd*noiswmd kickstart=/tmp/media/x86_64/install/alcasar/auto_inst.cfg.pl*g'  $EFI_GRUB_CFG 

# Generating distr
echo "Generating mirror tree..."
gendistrib -s ${MAGEIA_NEW_DIR}/${ARCH}

# Creating the new iso file
echo "Creating the isofile..."
newName=Mageia-$MAGEIA_VERSION-${ARCH}-Alcasar-$ALCASAR_VERSION
newIsoName=$newName.iso

# Generating the iso file
# Parameters for xorriso found using the following command on the Magiea iso file used in ressource folder
# ex: xorriso -indev Mageia-X.Y-x86_64.iso -report_el_torito as_mkisofs
cd ${MAGEIA_NEW_DIR} && xorriso -as mkisofs \
	-V $newName \
	--grub2-mbr --interval:local_fs:0s-15s:zero_mbrpt:$FULL_PATH_ISO_IN \
	-partition_cyl_align off \
	-partition_offset 0 \
	--mbr-force-bootable \
	-append_partition 2 0xef --interval:local_fs:8820924d-8829115d::$FULL_PATH_ISO_IN \
	-iso_mbr_part_type 0x00 \
	-c /boot.catalog \
	-b /boot/grub2/eltorito.img \
	-no-emul-boot \
	-boot-load-size 4 \
	-boot-info-table \
	--grub2-boot-info \
	-eltorito-alt-boot \
	-e '--interval:appended_partition_2_start_2205231s_size_8192d:all::' \
	-no-emul-boot \
	-boot-load-size 8192 \
	-o ${TMP_DIR}/${newIsoName} .

# Unmounting & Removing temporary dir
echo "Umounting..."
umount $MAGEIA_OFFICIAL_DIR
mv $coreDirNew/*.rpm $urpmi_cache
for directory in $MAGEIA_NEW_DIR $MAGEIA_OFFICIAL_DIR
do
	[ -d $directory ] && rm -rf $directory
done
rm -f "$AUTO_INSTALL_CFG"
echo "All RPM are always in $urpmi_cache. Remove them if nedded."
echo "The ISO file is here : ${TMP_DIR}/${newIsoName}"
