#!/bin/bash

#########################
## ALCASAR replication ##
##        start        ##
#########################
# The script is designed to start a connection to a remote.

# Constants
readonly ALCASAR_PWD="/root/ALCASAR-passwords.txt"

# Dynamically generated constants
DB_ROOT_PWD="$(grep db_root "$ALCASAR_PWD" | cut -d '=' -f 2-)"
readonly DB_ROOT_PWD;

# Variables
remote_to_start=""


# Check script args
# $@: script args
check_args() {
	# Parse args
	args="$(getopt --longoptions "all,name:,help" --options "a,n:,h" -- "$@")"

	# Reset script args list
	eval set -- "$args"

	# Print help
	if [ "$#" -eq 1 ]
	then
		usage
		return 1
	fi

	# Loop over all args
	while true
	do
		case "$1" in
			--all | -a)
				echo "Starting all connections."
				remote_to_start="ALL"
				break
				;;
			--name | -n)
				echo "Starting connection to $2."
				remote_to_start="$2"
				shift
				break
				;;
			--help | -h)
				usage
				return 2
				;;
			--)
				# End of args
				break
				;;
			*)
				echo "error: unknown $1" >&2
				return 3
				break
				;;
		esac
		shift
	done
}

# Execute SQL query
# $1: query
# $2: user (default: root)
# $3: password (default: root pwd)
# $4: host (default: localhost)
# $5: port (default: 3306)
exec_query() {
	# Check args
	if [ $# -lt 1 ]
	then
		echo "usage: $0 \"SQL query\" <DB user> <DB password> <SQL server address> <SQL server port>"
		return 2
	fi

	# Execute the query
	/usr/bin/mariadb --user="root" --password="$DB_ROOT_PWD" --execute="$1"
}

# Start all or specific replication with remotes
start_remote() {
	# Check args
	if [ $# -ne 1 ]
	then
		echo "error: need to pass remote name"
		return 1
	fi

	name="$1"
	service_file="/etc/systemd/system/replication-$name.service"

	# Skip if the connection is received (like on a primary)
	if [ -f "$service_file" ]
	then
		echo "Starting SSH tunnel with '$name'..."
		/usr/bin/systemctl start "replication-$name"
		/usr/bin/systemctl enable "replication-$name"

		# Get remote IP and port from its name
		ip="$(/usr/local/bin/alcasar-replication-list.sh --name=$name | grep "Master_Host" | cut -d ' ' -f 2-)"
		port="$(grep "ExecStart" "$service_file" | cut -d ' ' -f 9)"

		echo "Allowing outbound connection to remote SSH..."
		/usr/bin/sed -i -E "/^REPLICATION_TO=/s/=(.*)/=\1$ip:$port,/" /usr/local/etc/alcasar.conf
		/usr/local/bin/alcasar-iptables.sh
	fi

	echo "Starting replication with '$name'..."
	exec_query "START REPLICA '$name'" || return
}

# Print help message
usage() {
	echo "usage: $0 OPTIONS"
	echo
	echo "OPTIONS"
	echo "	--all, -a"
	echo "		start all connections"
	echo "	--name=NAME, -n NAME"
	echo "		remote name connection to start"
	echo "	--help, -h"
	echo "		print this help message"
}

# Main
check_args "$@" || exit

# Stop all or specific remote
if [ "$remote_to_start" == "ALL" ]
then
	# Get remotes names from database
	remotes_names=$(/usr/local/bin/alcasar-replication-list.sh --all | grep "Connection_name" | cut -d ':' -f 2-)

	# Loop over names
	for name in $remotes_names
	do
		start_remote "$name"
	done
else
	start_remote "$remote_to_start"
fi
